/**
* \file: am_api_events.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <sys/epoll.h>
#include <sys/eventfd.h>

#include "am_api_events.h"
#include "am_api_fsm.h"
#include "am_api_mutex.h"

#include "automounter_api_ctrl.h"

#include "utils/logger.h"
#include "ipc/message_recvr.h"
#include "ipc/request_messages.h"
#include "ipc/info_messages.h"

//------------------------------------- private attributes ------------------------------------------------------------
static const automounter_api_callbacks_t *am_api_events_callback_funcs;
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- private member declaration ----------------------------------------------------
static void am_api_events_on_request_done_msg_received(message_buffer_t *recvr_buffer);
static void am_api_events_on_snapshot_complete_msg_received(message_buffer_t *recvr_buffer);
static void am_api_events_on_devstate_changed_msg_received(message_buffer_t *recvr_buffer);
static void am_api_events_on_partstate_changed_msg_received(message_buffer_t *recvr_buffer);

static void am_api_events_call_constate_changed_callback(connection_state_changed_callback_t callback);
static void am_api_events_call_device_info_update_callback(const device_info_t *device_info, int request_id);
static void am_api_events_call_partition_info_update_callback(const partition_info_t *partition_info,
		const device_info_t *device_info, int request_id);
static void am_api_events_call_device_state_change_callback(device_state_changed_callback_t callback,
		const device_info_t *device_info);
static void am_api_events_call_partition_state_change_callback(partition_state_changed_callback_t callback,
		const partition_info_t *partition_info, const device_info_t *device_info);
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- public member definition ------------------------------------------------------
void am_api_events_signal_establish_connection_success(void)
{
	logger_log_debug("AUTOMOUNTER_API - Going to send a 'establish_connection_success' signal.");
	if (am_api_events_callback_funcs!=NULL)
		am_api_events_call_constate_changed_callback(am_api_events_callback_funcs->on_establish_connection_success);
	else
		logger_log_debug("AUTOMOUNTER_API - No callbacks registered with the shared library at all.");
}

void am_api_events_signal_establish_connection_failure(void)
{
	logger_log_debug("AUTOMOUNTER_API - Going to send a 'establish_connection_failed' signal.");
	if (am_api_events_callback_funcs!=NULL)
		am_api_events_call_constate_changed_callback(am_api_events_callback_funcs->on_establish_connection_failure);
	else
		logger_log_debug("AUTOMOUNTER_API - No callbacks registered with the shared library at all.");
}

void am_api_events_signal_connection_lost(void)
{
	logger_log_debug("AUTOMOUNTER_API - Going to send a 'connection_lost' signal.");
	if (am_api_events_callback_funcs!=NULL)
		am_api_events_call_constate_changed_callback(am_api_events_callback_funcs->on_connection_lost);
	else
		logger_log_debug("AUTOMOUNTER_API - No callbacks registered with the shared library at all.");
}

void am_api_events_register_callbacks(const automounter_api_callbacks_t *callback_funcs)
{
	am_api_events_callback_funcs=callback_funcs;
	am_api_fsm_signal_callbacks_changed();
}

void am_api_events_on_application_event_message_received(message_buffer_t *recvr_buffer)
{
	message_type_t msg_type=message_recvr_get_msg_type(recvr_buffer);
	logger_log_debug("AUTOMOUNTER_API - Received a message that results into an event send to"
			" the application (type=%d).",msg_type);
	if (msg_type==REQUEST_DONE_MSG)
		am_api_events_on_request_done_msg_received(recvr_buffer);
	else if (msg_type==SNAPSHOT_COMPLETE_MSG)
		am_api_events_on_snapshot_complete_msg_received(recvr_buffer);
	else if (msg_type==DEVICE_STATE_CHANGED_MSG)
		am_api_events_on_devstate_changed_msg_received(recvr_buffer);
	else if (msg_type==PARTITION_STATE_CHANGED_MSG)
		am_api_events_on_partstate_changed_msg_received(recvr_buffer);
	else
		logger_log_error("AUTOMOUNTER_API - Received an unknown application event message (type=%d). Ignoring it.",
				(int)msg_type);
}

event_mask_t am_api_events_get_callback_mask(void)
{
	event_mask_t event_mask=0;
	//nothing registered at all
	if (am_api_events_callback_funcs==NULL)
		return event_mask;

	if (am_api_events_callback_funcs->on_update_device_info!=NULL)
		event_mask |= (event_mask_t)DEVICE_UPDATE_INFO_EVENT;
	if (am_api_events_callback_funcs->on_device_detected!=NULL)
		event_mask |= (event_mask_t)DEVICE_DETECTED_EVENT;
	if (am_api_events_callback_funcs->on_device_nomedia!=NULL)
		event_mask |= (event_mask_t)DEVICE_NOMEDIA_EVENT;
	if (am_api_events_callback_funcs->on_device_automounting!=NULL)
		event_mask |= (event_mask_t)DEVICE_AUTOMOUNTING_EVENT;
	if (am_api_events_callback_funcs->on_device_automounted!=NULL)
		event_mask |= (event_mask_t)DEVICE_AUTOMOUNTED_EVENT;
	if (am_api_events_callback_funcs->on_device_unmounting!=NULL)
		event_mask |= (event_mask_t)DEVICE_UNMOUNTING_EVENT;
	if (am_api_events_callback_funcs->on_device_unmounted!=NULL)
		event_mask |= (event_mask_t)DEVICE_UNMOUNTED_EVENT;
	if (am_api_events_callback_funcs->on_device_invalid!=NULL)
		event_mask |= (event_mask_t)DEVICE_INVALID_EVENT;

	if (am_api_events_callback_funcs->on_update_partition_info!=NULL)
		event_mask |= (event_mask_t)PARTITION_UPDATE_INFO_EVENT;
	if (am_api_events_callback_funcs->on_partition_detected!=NULL)
		event_mask |= (event_mask_t)PARTITION_DETECTED_EVENT;
	if (am_api_events_callback_funcs->on_partition_unsupported!=NULL)
		event_mask |= (event_mask_t)PARTITION_UNSUPPORTED_EVENT;
	if (am_api_events_callback_funcs->on_partition_mounting!=NULL)
		event_mask |= (event_mask_t)PARTITION_MOUNTING_EVENT;
	if (am_api_events_callback_funcs->on_partition_mounted!=NULL)
		event_mask |= (event_mask_t)PARTITION_MOUNTED_EVENT;
	if (am_api_events_callback_funcs->on_partition_mount_err!=NULL)
		event_mask |= (event_mask_t)PARTITION_MOUNT_ERR_EVENT;
	if (am_api_events_callback_funcs->on_partition_remounting!=NULL)
		event_mask |= (event_mask_t)PARTITION_REMOUNTING_EVENT;
	if (am_api_events_callback_funcs->on_partition_unmounting!=NULL)
		event_mask |= (event_mask_t)PARTITION_UNMOUNTING_EVENT;
	if (am_api_events_callback_funcs->on_partition_unmounted!=NULL)
		event_mask |= (event_mask_t)PARTITION_UNMOUNTED_EVENT;
	if (am_api_events_callback_funcs->on_partition_invalid!=NULL)
		event_mask |= (event_mask_t)PARTITION_INVALID_EVENT;

	return event_mask;
}
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- private member definition -----------------------------------------------------
static void am_api_events_call_constate_changed_callback(connection_state_changed_callback_t callback)
{
	if (callback!=NULL)
	{
		//we don't need to take care for any library internal data since we are not using any
		am_api_mutex_leaving_library();
		callback();
		am_api_mutex_entering_library();
	}
	else
		logger_log_debug("AUTOMOUNTER_API - No one registered for the signal.");
}

static void am_api_events_call_device_info_update_callback(const device_info_t *device_info, int request_id)
{
	update_device_info_callback_t callback;
	if (am_api_events_callback_funcs->on_update_device_info==NULL)
	{
		logger_log_debug("AUTOMOUNTER_API - No one registered for the signal.");
		return;
	}

	// we take a copy of the callback function here since we want to unlock the library mutex. This opens a way for
	// others to modify the callback function structure.
	callback=am_api_events_callback_funcs->on_update_device_info;

	// we can easily unlock the library mutex here. The data we are using is secured as follows:
	// - am_api_events_callback_funcs->on_snapshot_complete: We took a copy
	// - device_info: Points to the message received from the automounter. This message is
	//	 				   stored in the receiver buffer, which is protected by the event_dispatcher_mutex
	//					   which is locked in:
	//						- both dispatcher functions (rw access of the message buffer)
	//						- the init / deinit function of the library that sets up and destroys the buffer
	//					   The data item is not accessed from any other point.
	am_api_mutex_leaving_library();
	callback(device_info,request_id);
	am_api_mutex_entering_library();
	// at this point, the internal state of the library might have been changed by others. This is not a problem
	// since we are just returning back from all functions that brought us here without doing anything anymore
}

static void am_api_events_call_partition_info_update_callback(const partition_info_t *partition_info,
		const device_info_t *device_info, int request_id)
{
	update_partition_info_callback_t callback;
	if (am_api_events_callback_funcs->on_update_partition_info==NULL)
	{
		logger_log_debug("AUTOMOUNTER_API - No one registered for the signal.");
		return;
	}

	// we take a copy of the callback function here since we want to unlock the library mutex. This opens a way for
	// others to modify the callback function structure.
	callback=am_api_events_callback_funcs->on_update_partition_info;

	// we can easily unlock the library mutex here. The data we are using is secured as follows:
	// - am_api_events_callback_funcs->on_snapshot_complete: We took a copy
	// - partition_info, device_info: Points to the message received from the automounter. This message is
	//	 				   stored in the receiver buffer, which is protected by the event_dispatcher_mutex
	//					   which is locked in:
	//						- both dispatcher functions (rw access of the message buffer)
	//						- the init / deinit function of the library that sets up and destroys the buffer
	//					   The data item is not accessed from any other point.
	am_api_mutex_leaving_library();
	callback(partition_info,device_info,request_id);
	am_api_mutex_entering_library();
	// at this point, the internal state of the library might have been changed by others. This is not a problem
	// since we are just returning back from all functions that brought us here without doing anything anymore
}

static void am_api_events_call_device_state_change_callback(device_state_changed_callback_t callback,
		const device_info_t *device_info)
{
	if (callback==NULL)
	{
		logger_log_debug("AUTOMOUNTER_API - No one registered for the signal.");
		return;
	}

	//we already took already a copy of the callback function (we passed it as parameter)

	// we can easily unlock the library mutex here. The data we are using is secured as follows:
	// - device_info: Points to the message received from the automounter. This message is
	//	 				   stored in the receiver buffer, which is protected by the event_dispatcher_mutex
	//					   which is locked in:
	//						- both dispatcher functions (rw access of the message buffer)
	//						- the init / deinit function of the library that sets up and destroys the buffer
	//					   The data item is not accessed from any other point.
	am_api_mutex_leaving_library();
	callback(device_info);
	am_api_mutex_entering_library();
	// at this point, the internal state of the library might have been changed by others. This is not a problem
	// since we are just returning back from all functions that brought us here without doing anything anymore
}

static void am_api_events_call_partition_state_change_callback(partition_state_changed_callback_t callback,
		const partition_info_t *partition_info, const device_info_t *device_info)
{
	if (callback==NULL)
	{
		logger_log_debug("AUTOMOUNTER_API - No one registered for the signal.");
		return;
	}

	//we already took already a copy of the callback function (we passed it as parameter)

	// we can easily unlock the library mutex here. The data we are using is secured as follows:
	// - partition_info, device_info: Points to the message received from the automounter. This message is
	//	 				   stored in the receiver buffer, which is protected by the event_dispatcher_mutex
	//					   which is locked in:
	//						- both dispatcher functions (rw access of the message buffer)
	//						- the init / deinit function of the library that sets up and destroys the buffer
	//					   The data item is not accessed from any other point.
	am_api_mutex_leaving_library();
	callback(partition_info,device_info);
	am_api_mutex_entering_library();
	// at this point, the internal state of the library might have been changed by others. This is not a problem
	// since we are just returning back from all functions that brought us here without doing anything anymore
}

static void am_api_events_on_request_done_msg_received(message_buffer_t *recvr_buffer)
{
	request_done_message_t *request_done_msg;
	request_done_callback_t callback_func;

	if (request_message_extract_request_done_msg(recvr_buffer,&request_done_msg)!=RESULT_OK)
	{
		logger_log_error("AUTOMOUNTER_API - Received a corrupt 'request_done' message. Ignoring it.");
		return;
	}

	callback_func=(request_done_callback_t)request_done_msg->callback_func;
	if (callback_func!=NULL)
	{
		logger_log_debug("AUTOMOUNTER_API - Going to call the 'request_done' callback.");
		// we can easily unlock the library mutex here. The data we are using is secured as follows:
		// - request_done_msg: This data structure contains the message received from the automounter. This message is
		//	 				   stored in the receiver buffer which is protected by the event_dispatcher_mutex
		//					   which is locked in:
		//						- both dispatcher functions (rw access of the message buffer)
		//						- the init / deinit function of the library that sets up and destroys the buffer
		//					   The data item is not accessed from any other point.
		am_api_mutex_leaving_library();
		callback_func(request_done_msg->request_id,	request_done_msg->result,request_done_msg->error_message);
		am_api_mutex_entering_library();
		// at this point, the internal state of the library might have been changed by others. This is not a problem
		// since we are just returning back from all functions that brought us here without doing anything anymore
	}
}

static void am_api_events_on_snapshot_complete_msg_received(message_buffer_t *recvr_buffer)
{
	int *request_id_ptr;
	snapshot_complete_callback_t callback;

	if (request_message_extract_snapshot_done_msg(recvr_buffer,&request_id_ptr)!=RESULT_OK)
	{
		logger_log_error("AUTOMOUNTER_API - Received a corrupt 'snapshot_complete' message. Ignoring it.");
		return;
	}

	logger_log_debug("AUTOMOUNTER_API - Going to send a 'snapshot complete' signal.");

	if (am_api_events_callback_funcs==NULL)
	{
		logger_log_debug("AUTOMOUNTER_API - No callbacks registered with the shared library at all.");
		return;
	}

	if (am_api_events_callback_funcs->on_snapshot_complete!=NULL)
	{
		// we take a copy of the callback function here since we want to unlock the library mutex. This opens a way for
		// others to modify the callback function structure.
		callback=am_api_events_callback_funcs->on_snapshot_complete;

		// we can easily unlock the library mutex here. The data we are using is secured as follows:
		// - am_api_events_callback_funcs->on_snapshot_complete: We took a copy
		// - request_id_ptr: Points to the message received from the automounter. This message is
		//	 				   stored in the receiver buffer, which is protected by the event_dispatcher_mutex
		//					   which is locked in:
		//						- both dispatcher functions (rw access of the message buffer)
		//						- the init / deinit function of the library that sets up and destroys the buffer
		//					   The data item is not accessed from any other point.
		am_api_mutex_leaving_library();
		callback(*request_id_ptr);
		am_api_mutex_entering_library();
		// at this point, the internal state of the library might have been changed by others. This is not a problem
		// since we are just returning back from all functions that brought us here without doing anything anymore
	}
	else
		logger_log_debug("AUTOMOUNTER_API - No one registered for the signal.");
}

static void am_api_events_on_devstate_changed_msg_received(message_buffer_t *recvr_buffer)
{
	device_info_message_t *dev_info_msg;
	if (info_messages_extract_dev_info_msg(recvr_buffer,&dev_info_msg)!=RESULT_OK)
	{
		logger_log_error("AUTOMOUNTER_API - Received a corrupt 'dev_info' message. Ignoring it.");
		return;
	}

	if (dev_info_msg->event!=DEVICE_UPDATE_INFO_EVENT)
		logger_log_debug("AUTOMOUNTER_API - Going to send a 'device state changed' signal (state change event: 0x%X).",
			(int)dev_info_msg->event);
	else
		logger_log_debug("AUTOMOUNTER_API - Going to send a 'device update info' signal (request_id=%d).",
				dev_info_msg->request_id);

	if (am_api_events_callback_funcs==NULL)
	{
		logger_log_debug("AUTOMOUNTER_API - No callbacks registered with the shared library at all.");
		return;
	}

	switch(dev_info_msg->event)
	{
	case DEVICE_UPDATE_INFO_EVENT:
		am_api_events_call_device_info_update_callback(&(dev_info_msg->device_info),dev_info_msg->request_id);
		break;
	case DEVICE_DETECTED_EVENT:
		am_api_events_call_device_state_change_callback(am_api_events_callback_funcs->on_device_detected,
				&(dev_info_msg->device_info));
		break;
	case DEVICE_NOMEDIA_EVENT:
		am_api_events_call_device_state_change_callback(am_api_events_callback_funcs->on_device_nomedia,
				&(dev_info_msg->device_info));
		break;
	case DEVICE_AUTOMOUNTING_EVENT:
		am_api_events_call_device_state_change_callback(am_api_events_callback_funcs->on_device_automounting,
				&(dev_info_msg->device_info));
		break;
	case DEVICE_AUTOMOUNTED_EVENT:
		am_api_events_call_device_state_change_callback(am_api_events_callback_funcs->on_device_automounted,
				&(dev_info_msg->device_info));
		break;
	case DEVICE_UNMOUNTING_EVENT:
		am_api_events_call_device_state_change_callback(am_api_events_callback_funcs->on_device_unmounting,
				&(dev_info_msg->device_info));
		break;
	case DEVICE_UNMOUNTED_EVENT:
		am_api_events_call_device_state_change_callback(am_api_events_callback_funcs->on_device_unmounted,
				&(dev_info_msg->device_info));
		break;
	case DEVICE_INVALID_EVENT:
		am_api_events_call_device_state_change_callback(am_api_events_callback_funcs->on_device_invalid,
				&(dev_info_msg->device_info));
		break;
	case PARTITION_UPDATE_INFO_EVENT:
	case PARTITION_DETECTED_EVENT:
	case PARTITION_UNSUPPORTED_EVENT:
	case PARTITION_MOUNTING_EVENT:
	case PARTITION_MOUNTED_EVENT:
	case PARTITION_MOUNT_ERR_EVENT:
	case PARTITION_REMOUNTING_EVENT:
	case PARTITION_UNMOUNTING_EVENT:
	case PARTITION_UNMOUNTED_EVENT:
	case PARTITION_INVALID_EVENT:
	case SNAP_SHOT_DONE_EVENT:
	default:
		logger_log_error("AUTOMOUNTER_API - Received unknown device state change event(0x%X). Ignoring it.",
				(int)dev_info_msg->event);
		break;
	}
}

static void am_api_events_on_partstate_changed_msg_received(message_buffer_t *recvr_buffer)
{
	partition_info_message_t *part_info_msg;
	if (info_messages_extract_part_info_msg(recvr_buffer,&part_info_msg)!=RESULT_OK)
	{
		logger_log_error("AUTOMOUNTER_API - Received a corrupt 'part_info' message. Ignoring it.");
		return;
	}

	if (part_info_msg->event!=PARTITION_UPDATE_INFO_EVENT)
		logger_log_debug("AUTOMOUNTER_API - Going to send a 'partition state changed' signal (state change event: 0x%X).",
			part_info_msg->event);
	else
		logger_log_debug("AUTOMOUNTER_API - Going to send a 'partition update info' signal request_id: %d).",
			part_info_msg->request_id);

	if (am_api_events_callback_funcs==NULL)
	{
		logger_log_debug("AUTOMOUNTER_API - No callbacks registered with the shared library at all.");
		return;
	}

	switch(part_info_msg->event)
	{
	case PARTITION_UPDATE_INFO_EVENT:
		am_api_events_call_partition_info_update_callback(&(part_info_msg->part_info), &(part_info_msg->device_info),
				part_info_msg->request_id);
		break;
	case PARTITION_DETECTED_EVENT:
		am_api_events_call_partition_state_change_callback(am_api_events_callback_funcs->on_partition_detected,
				&(part_info_msg->part_info),&(part_info_msg->device_info));
		break;
	case PARTITION_UNSUPPORTED_EVENT:
		am_api_events_call_partition_state_change_callback(am_api_events_callback_funcs->on_partition_unsupported,
				&(part_info_msg->part_info),&(part_info_msg->device_info));
		break;
	case PARTITION_MOUNTING_EVENT:
		am_api_events_call_partition_state_change_callback(am_api_events_callback_funcs->on_partition_mounting,
				&(part_info_msg->part_info),&(part_info_msg->device_info));
		break;
	case PARTITION_MOUNTED_EVENT:
		am_api_events_call_partition_state_change_callback(am_api_events_callback_funcs->on_partition_mounted,
				&(part_info_msg->part_info),&(part_info_msg->device_info));
		break;
	case PARTITION_MOUNT_ERR_EVENT:
		am_api_events_call_partition_state_change_callback(am_api_events_callback_funcs->on_partition_mount_err,
				&(part_info_msg->part_info),&(part_info_msg->device_info));
		break;
	case PARTITION_REMOUNTING_EVENT:
		am_api_events_call_partition_state_change_callback(am_api_events_callback_funcs->on_partition_remounting,
				&(part_info_msg->part_info),&(part_info_msg->device_info));
		break;
	case PARTITION_UNMOUNTING_EVENT:
		am_api_events_call_partition_state_change_callback(am_api_events_callback_funcs->on_partition_unmounting,
				&(part_info_msg->part_info),&(part_info_msg->device_info));
		break;
	case PARTITION_UNMOUNTED_EVENT:
		am_api_events_call_partition_state_change_callback(am_api_events_callback_funcs->on_partition_unmounted,
				&(part_info_msg->part_info),&(part_info_msg->device_info));
		break;
	case PARTITION_INVALID_EVENT:
		am_api_events_call_partition_state_change_callback(am_api_events_callback_funcs->on_partition_invalid,
				&(part_info_msg->part_info),&(part_info_msg->device_info));
		break;
	case DEVICE_UPDATE_INFO_EVENT:
	case DEVICE_DETECTED_EVENT:
	case DEVICE_NOMEDIA_EVENT:
	case DEVICE_AUTOMOUNTING_EVENT:
	case DEVICE_AUTOMOUNTED_EVENT:
	case DEVICE_UNMOUNTING_EVENT:
	case DEVICE_UNMOUNTED_EVENT:
	case DEVICE_INVALID_EVENT:
	case SNAP_SHOT_DONE_EVENT:
	default:
		logger_log_error("AUTOMOUNTER_API - Received unknown partition state change event (0x%X). Ignoring it.",
				(int)part_info_msg->event);
		break;
	}
}
//---------------------------------------------------------------------------------------------------------------------
